<?php

namespace App\Livewire;

use Livewire\Component;
use App\Services\AnnouncementService;
use App\Traits\HasNotifications;
use App\Models\Announcement;

class AnnouncementForm extends Component
{
    use HasNotifications;

    public $title = '';
    public $message = '';
    public $type = 'popup';
    public $alert_type = 'info';
    public $target_type = 'all';
    public $target_id = null;
    public $display_from = '';
    public $display_until = '';
    public $active = true;

    public $announcement = null;
    public $isEditing = false;

    protected $announcementService;

    public function boot(AnnouncementService $announcementService)
    {
        $this->announcementService = $announcementService;
    }

    public function mount($announcementId = null)
    {
        if ($announcementId) {
            $this->announcement = Announcement::where('announcement_id', $announcementId)->firstOrFail();
            $this->isEditing = true;
            $this->fill($this->announcement->toArray());
            $this->display_from = $this->announcement->display_from?->format('Y-m-d\TH:i');
            $this->display_until = $this->announcement->display_until?->format('Y-m-d\TH:i');
        }
    }

    public function rules()
    {
        return [
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:popup,fixed',
            'alert_type' => 'required|in:success,warning,info,error',
            'target_type' => 'required|in:all,all_students,all_staff,staff,student,class,batch',
            'target_id' => 'nullable|integer',
            'display_from' => 'nullable|date',
            'display_until' => 'nullable|date|after:display_from',
            'active' => 'boolean',
        ];
    }

    public function updatedTargetType()
    {
        $this->target_id = null;
    }

    public function save()
    {
        $this->validate();

        $data = [
            'title' => $this->title,
            'message' => $this->message,
            'type' => $this->type,
            'alert_type' => $this->alert_type,
            'target_type' => $this->target_type,
            'target_id' => $this->target_id,
            'display_from' => $this->display_from ? \Carbon\Carbon::parse($this->display_from) : null,
            'display_until' => $this->display_until ? \Carbon\Carbon::parse($this->display_until) : null,
            'active' => $this->active,
            'created_by' => auth()->id(),
        ];

        if ($this->isEditing) {
            $this->announcementService->updateAnnouncement($this->announcement, $data);
            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Announcement updated successfully!'
            ]);
        } else {
            $this->announcementService->createAnnouncement($data);
            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Announcement created successfully!'
            ]);
            $this->reset();
        }

        return redirect()->route('announcements.index');
    }

    public function render()
    {
        $targetOptions = $this->announcementService->getTargetOptions();
        $students = $this->target_type === 'student' ? $this->announcementService->getStudents() : collect();
        $staff = $this->target_type === 'staff' ? $this->announcementService->getStaff() : collect();
        $classes = $this->target_type === 'class' ? $this->announcementService->getClasses() : collect();
        $batches = $this->target_type === 'batch' ? $this->announcementService->getBatches() : collect();

        return view('livewire.announcement-form', compact('targetOptions', 'students', 'staff', 'classes', 'batches'));
    }
}