<?php

namespace App\Livewire;

use App\Models\Batch;
use App\Models\SchoolClass;
use App\Services\BatchService;
use App\Traits\HasNotifications;
use Livewire\Component;

class BatchForm extends Component
{
    use HasNotifications;

    public $batchId;
    public $name = '';
    public $school_class_id = '';
    public $description = '';
    public $capacity = 30;
    public $is_active = true;

    protected $rules = [
        'name' => 'required|string|max:255',
        'school_class_id' => 'required|exists:school_classes,id',
        'description' => 'nullable|string',
        'capacity' => 'required|integer|min:1|max:100',
        'is_active' => 'boolean'
    ];

    public function mount($batchId = null)
    {
        if ($batchId) {
            $batch = Batch::findOrFail($batchId);
            $this->batchId = $batch->id;
            $this->name = $batch->name;
            $this->school_class_id = $batch->school_class_id;
            $this->description = $batch->description;
            $this->capacity = $batch->capacity;
            $this->is_active = $batch->is_active;
        }
    }

    public function save(BatchService $batchService)
    {
        $this->validate();

        try {
            if ($this->batchId) {
                $batch = Batch::findOrFail($this->batchId);
                $batchService->update($batch, [
                    'name' => $this->name,
                    'school_class_id' => $this->school_class_id,
                    'description' => $this->description,
                    'capacity' => $this->capacity,
                    'is_active' => $this->is_active
                ]);
                session()->flash('toast', ['type' => 'success', 'message' => 'Batch updated successfully!']);
            } else {
                $batchService->create([
                    'name' => $this->name,
                    'school_class_id' => $this->school_class_id,
                    'description' => $this->description,
                    'capacity' => $this->capacity,
                    'is_active' => $this->is_active
                ]);
                session()->flash('toast', ['type' => 'success', 'message' => 'Batch created successfully!']);
            }

            return redirect()->route('batches.index');
        } catch (\Exception $e) {
            $this->showToast('error', 'An error occurred while saving the batch.');
        }
    }

    public function render()
    {
        $schoolClasses = SchoolClass::where('is_active', true)->orderBy('order')->get();
        return view('livewire.batch-form', compact('schoolClasses'));
    }
}
