<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\User;
use App\Models\Payment;
use App\Models\PaymentRequest;
use App\Models\Wallet;
use App\Traits\HasNotifications;

class CreatePaymentForm extends Component
{
    use HasNotifications;

    public $student_search = '';
    public $selected_student_id = '';
    public $selected_student_name = '';
    public $payment_request_id = '';
    public $amount = '';
    public $currency = 'NGN';
    public $notes = '';
    public $status = 'successful';

    public $search_results = [];
    public $show_search_results = false;
    public $payment_requests = [];

    protected $rules = [
        'selected_student_id' => 'required|exists:users,id',
        'amount' => 'required|numeric|min:0.01',
        'currency' => 'required|string|size:3',
        'notes' => 'required|string|max:500',
        'status' => 'required|in:successful,pending,failed',
        'payment_request_id' => 'nullable|exists:payment_requests,id',
    ];

    public function updatedStudentSearch()
    {
        if (strlen($this->student_search) >= 2) {
            $this->search_results = User::whereNull('deleted_at')
                ->whereDoesntHave('roles')
                ->where(function($query) {
                    $query->where('firstname', 'like', '%' . $this->student_search . '%')
                          ->orWhere('middlename', 'like', '%' . $this->student_search . '%')
                          ->orWhere('surname', 'like', '%' . $this->student_search . '%')
                          ->orWhere('student_id', 'like', '%' . $this->student_search . '%');
                })
                ->with('batch.schoolClass')
                ->orderBy('firstname')
                ->limit(20)
                ->get(['id', 'firstname', 'middlename', 'surname', 'student_id', 'batch_id']);
            $this->show_search_results = true;
        } else {
            $this->search_results = [];
            $this->show_search_results = false;
        }
    }

    public function selectStudent($studentId)
    {
        $student = User::with('batch.schoolClass')->find($studentId);
        if ($student) {
            $this->selected_student_id = $student->id;
            $this->selected_student_name = $student->name;
            $this->student_search = $student->name;
            $this->show_search_results = false;
            
            // Load payment requests for this student
            $this->loadPaymentRequests();
        }
    }

    public function updatedPaymentRequestId()
    {
        if ($this->payment_request_id) {
            $paymentRequest = PaymentRequest::find($this->payment_request_id);
            if ($paymentRequest) {
                $this->amount = $paymentRequest->amount;
                $this->currency = $paymentRequest->currency;
            }
        } else {
            $this->amount = '';
            $this->currency = 'NGN';
        }
    }

    public function loadPaymentRequests()
    {
        if ($this->selected_student_id) {
            $student = User::find($this->selected_student_id);
            $this->payment_requests = PaymentRequest::where('status', 'active')
                ->where(function($query) use ($student) {
                    $query->where(function($q) use ($student) {
                        $q->where('target_type', 'student')
                          ->where('student_id', $student->id);
                    })
                    ->orWhere(function($q) use ($student) {
                        $q->where('target_type', 'class')
                          ->where('school_class_id', $student->school_class_id);
                    })
                    ->orWhere(function($q) use ($student) {
                        $q->where('target_type', 'batch')
                          ->where('batch_id', $student->batch_id);
                    })
                    ->orWhere('target_type', 'all_students');
                })
                ->orderBy('created_at', 'desc')
                ->get(['id', 'title', 'amount', 'currency']);
        }
    }

    public function save()
    {
        $this->validate();

        try {
            $paymentData = [
                'payment_id' => $this->generateUniquePaymentId(),
                'user_id' => $this->selected_student_id,
                'amount' => $this->amount,
                'currency' => $this->currency,
                'status' => $this->status,
                'notes' => $this->notes,
                'payment_date' => now(),
                'recorded_by' => auth()->id(),
            ];

            if ($this->payment_request_id) {
                $paymentData['payment_request_id'] = $this->payment_request_id;
                
                // If successful and linked to payment request, update payment request
                if ($this->status === 'successful') {
                    $paymentRequest = PaymentRequest::find($this->payment_request_id);
                    $paymentRequest->increment('amount_paid', $this->amount);
                    $paymentRequest->update(['balance' => $paymentRequest->amount - $paymentRequest->amount_paid]);
                }
            }

            Payment::create($paymentData);
            
            // Credit wallet if payment is successful
            if ($this->status === 'successful') {
                Wallet::creditWallet($this->currency, $this->amount);
            }

            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Payment created successfully!'
            ]);

            return redirect()->route('payments.history');
        } catch (\Exception $e) {
            $this->showToast('error', 'Error creating payment: ' . $e->getMessage(), 'Payment');
        }
    }

    private function generateUniquePaymentId()
    {
        do {
            $id = strtoupper(substr(str_shuffle('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 10));
        } while (Payment::where('payment_id', $id)->exists());
        
        return $id;
    }

    public function render()
    {
        return view('livewire.create-payment-form');
    }
}