<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Payment;
use App\Models\PaymentRequest;
use App\Models\Wallet;
use App\Traits\HasNotifications;

class PaymentStatusForm extends Component
{
    use HasNotifications;

    public Payment $payment;
    public $status;
    public $notes = '';
    public $partialAmount = '';
    public $balance = 0;
    public $editingNotes = false;
    
    protected $listeners = ['action-confirmed' => 'handleConfirmedAction'];
    
    public function getCanUpdateStatusProperty()
    {
        return in_array($this->payment->status, ['pending']);
    }

    public function mount($paymentId)
    {
        $this->payment = Payment::where('payment_id', $paymentId)->with(['paymentRequest', 'user'])->firstOrFail();
        $this->status = $this->payment->status;
        $this->notes = $this->payment->notes ?? '';
    }

    public function updatedStatus()
    {
        if ($this->status !== 'partial') {
            $this->partialAmount = '';
            $this->balance = 0;
        }
    }

    public function updatedPartialAmount()
    {
        if ($this->partialAmount && is_numeric($this->partialAmount)) {
            $this->balance = max(0, $this->payment->amount - $this->partialAmount);
        } else {
            $this->balance = 0;
        }
    }

    public function confirmUpdate()
    {
        $message = 'Are you sure you want to update this payment status?';
        if ($this->status === 'partial' && $this->partialAmount) {
            $message .= ' A new payment request will be created for the balance amount.';
        }
        
        $this->confirmAction('Update Payment Status', $message, 'Yes, Update!');
    }
    
    public function handleConfirmedAction()
    {
        $this->updateStatus();
    }
    
    public function enableNotesEditing()
    {
        $this->editingNotes = true;
    }
    
    public function updateNotes()
    {
        $this->validate(['notes' => 'nullable|string']);
        
        $this->payment->update([
            'notes' => $this->notes,
            'recorded_by' => auth()->id(),
        ]);
        
        $this->editingNotes = false;
        $this->showToast('success', 'Notes updated successfully!', 'Payment');
    }

    public function updateStatus()
    {
        try {
            
            if ($this->status === 'partial') {
                $this->validate([
                    'status' => 'required|in:successful,partial,failed,cancelled,pending',
                    'notes' => 'nullable|string',
                    'partialAmount' => 'required|numeric|min:0.01|max:' . $this->payment->amount,
                ]);
            } else {
                $this->validate([
                    'status' => 'required|in:successful,partial,failed,cancelled,pending',
                    'notes' => 'nullable|string',
                ]);
            }
            if ($this->status === 'partial' && $this->partialAmount) {
                $balance = $this->payment->amount - $this->partialAmount;
                
                // Update payment to successful and create new payment request for balance
                $updateData = [
                    'status' => 'successful',
                    'amount' => $this->partialAmount,
                    'notes' => $this->notes,
                    'recorded_by' => auth()->id(),
                ];
                
                if ($balance > 0) {
                    $originalRequest = $this->payment->paymentRequest;
                    
                    PaymentRequest::create([
                        'title' => 'Balance Payment - ' . $originalRequest->title,
                        'description' => 'Balance on partial payment for: ' . $originalRequest->title,
                        'amount' => $balance,
                        'currency' => $this->payment->currency,
                        'target_type' => 'student',
                        'student_id' => $this->payment->user_id,
                        'created_by' => auth()->id(),
                    ]);
                }
            } else {
                $updateData = [
                    'status' => $this->status,
                    'notes' => $this->notes,
                    'recorded_by' => auth()->id(),
                ];
            }

            $this->payment->update($updateData);
            
            // Update payment request amount_paid when payment is marked as successful
            if ($this->status === 'successful') {
                $paymentRequest = $this->payment->paymentRequest;
                $paymentRequest->increment('amount_paid', $this->payment->amount);
                $paymentRequest->update(['balance' => $paymentRequest->amount - $paymentRequest->amount_paid]);
                
                // Credit wallet
                Wallet::creditWallet($this->payment->currency, $this->payment->amount);
            }

            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Payment status updated successfully!' . ($this->status === 'partial' && $this->balance > 0 ? ' Payment marked as successful and balance payment request created.' : '')
            ]);

            $this->redirect(route('payments.history'));
        } catch (\Exception $e) {
            $this->showToast('error', 'Error updating payment status: ' . $e->getMessage(), 'Payment');
        }
    }

    public function render()
    {
        return view('livewire.payment-status-form');
    }
}
