<?php

namespace App\Livewire;

use Livewire\Component;
use App\Services\PaymentService;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\Batch;
use App\Traits\HasNotifications;

class PaymentsByTarget extends Component
{
    use HasNotifications;

    public $target_type = 'student';
    public $target_id = '';
    public $paymentRequests = [];
    public $targetName = '';

    public $students = [];
    public $classes = [];
    public $batches = [];

    public function mount()
    {
        $this->loadData();
    }

    public function updatedTargetType()
    {
        $this->target_id = '';
        $this->paymentRequests = [];
        $this->targetName = '';
        
        if ($this->target_type === 'all_students') {
            $this->loadPayments();
        }
    }

    public function updatedTargetId()
    {
        if ($this->target_id || $this->target_type === 'all_students') {
            $this->loadPayments();
        } else {
            $this->paymentRequests = [];
            $this->targetName = '';
        }
    }

    public function loadData()
    {
        $this->students = User::whereNull('deleted_at')
            ->whereDoesntHave('roles')
            ->orderBy('firstname')
            ->get(['id', 'firstname', 'surname']);
            
        $this->classes = SchoolClass::orderBy('order')->get(['id', 'name', 'alias']);
        $this->batches = Batch::with('schoolClass')->orderBy('name')->get(['id', 'name', 'school_class_id']);
    }

    public function loadPayments()
    {
        if ($this->target_type === 'all_students' || ($this->target_id && $this->target_type)) {
            try {
                $paymentService = new PaymentService();
                $this->paymentRequests = $paymentService->getPaymentRequestsForTarget($this->target_type, $this->target_id);
                
                // Set target name
                switch ($this->target_type) {
                    case 'all_students':
                        $this->targetName = 'All Students';
                        break;
                    case 'student':
                        $student = User::find($this->target_id);
                        $this->targetName = $student ? $student->name : 'Unknown Student';
                        break;
                    case 'class':
                        $class = SchoolClass::find($this->target_id);
                        $this->targetName = $class ? $class->name : 'Unknown Class';
                        break;
                    case 'batch':
                        $batch = Batch::find($this->target_id);
                        $this->targetName = $batch ? $batch->name : 'Unknown Batch';
                        break;
                }
            } catch (\Exception $e) {
                session()->flash('toast', [
                    'type' => 'error',
                    'message' => 'Error loading payments: ' . $e->getMessage()
                ]);
            }
        }
    }

    public function render()
    {
        return view('livewire.payments-by-target');
    }
}
