<?php

namespace App\Livewire\Results;

use Livewire\Component;
use App\Models\User;
use App\Models\SchoolSession;
use App\Models\SchoolClass;
use App\Models\AnnualResultApproval;
use App\Services\ResultService;
use App\Services\AnnualResultService;
use App\Traits\HasNotifications;
use Illuminate\Support\Facades\Auth;

class AnnualApprovalForm extends Component
{
    use HasNotifications;

    public $selectedClass = '';
    public $selectedBatch = '';
    public $selectedStudents = [];
    public $selectedYear = '';
    public $classes = [];
    public $batches = [];
    public $students = [];
    public $years = [];
    public $studentResults = [];

    // Affective Traits
    public $punctuality = 4.0;
    public $mental_alertness = 4.0;
    public $attentiveness = 4.0;
    public $respect = 4.0;
    public $neatness = 4.0;
    public $politeness = 4.0;
    public $honesty = 4.0;
    public $relationship_with_peers = 4.0;
    public $attitude_to_school = 4.0;
    public $teamwork = 4.0;
    public $completes_school_work_promptly = 4.0;

    // Psychomotor Skills
    public $reading = 4.0;
    public $verbal_fluency_diction = 4.0;
    public $handwriting = 4.0;
    public $musical_skills = 4.0;
    public $creative_arts = 4.0;
    public $physical_education = 4.0;
    public $general_reasoning = 4.0;

    public $principal_remarks = 'Impressive result, keep it up!';

    public function mount()
    {
        $user = Auth::user();

        // Check if user can approve results based on settings
        if (!$user->canApproveResults()) {
            abort(403, 'You do not have permission to approve results.');
        }

        if ($user->isAdmin() || $user->isHeadTeacher()) {
            $this->classes = SchoolClass::orderBy('order')->get();
        } elseif ($user->isTeacher()) {
            $this->classes = $user->classes()->orderBy('order')->get();
        }

        // Get available years from all sessions
        $this->years = SchoolSession::distinct()
            ->orderBy('start_year', 'desc')
            ->pluck('start_year')
            ->toArray();

        // Auto-select currently active session year
        $activeSession = SchoolSession::where('is_active', true)->first();
        if ($activeSession && in_array($activeSession->start_year, $this->years)) {
            $this->selectedYear = $activeSession->start_year;
        } elseif (!empty($this->years)) {
            $this->selectedYear = $this->years[0]; // Select the most recent year if active session year not found
        }
    }

    public function updatedSelectedYear()
    {
        $this->selectedClass = '';
        $this->selectedBatch = '';
        $this->batches = [];
        $this->students = [];
        $this->selectedStudents = [];
        $this->studentResults = [];
    }

    public function updatedSelectedClass()
    {
        $this->selectedBatch = '';
        $this->batches = [];
        $this->students = [];
        $this->selectedStudents = [];
        $this->studentResults = [];

        if ($this->selectedClass) {
            $this->loadBatches();
        }
    }

    public function loadBatches()
    {
        $this->batches = \App\Models\Batch::where('school_class_id', $this->selectedClass)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();
    }

    public function updatedSelectedBatch()
    {
        $this->students = [];
        $this->selectedStudents = [];
        $this->studentResults = [];

        if ($this->selectedBatch && $this->selectedYear) {
            $resultService = new ResultService();
            $allStudents = $resultService->getStudentsByBatch($this->selectedBatch)->sortBy('firstname');

            // Exclude students with already approved annual results for this year
            $this->students = $allStudents->filter(function ($student) {
                $approvedResults = AnnualResultApproval::where('student_id', $student->id)
                    ->where('year', $this->selectedYear)
                    ->exists();
                return !$approvedResults;
            })->values()->all();
        }
    }

    public function updatedSelectedStudents()
    {
        $this->studentResults = [];
    }

    public function loadStudentResults()
    {
        $this->studentResults = [];

        if (!$this->selectedClass || !$this->selectedYear) return;

        $annualResultService = new AnnualResultService();

        foreach ($this->selectedStudents as $studentId) {
            $student = User::find($studentId);
            if (!$student) continue;

            $annualData = $annualResultService->getStudentAnnualData($studentId, $this->selectedYear);

            if ($annualData) {
                $this->studentResults[$studentId] = [
                    'student' => $student,
                    'sessions' => $annualData['sessions'],
                    'subjects' => $annualData['subjects'],
                    'annual_total' => $annualData['annual_total'],
                    'annual_average' => $annualData['annual_average'],
                    'sessions_totals' => $annualData['sessions_totals']
                ];
            }
        }
    }

    public function confirmApproval()
    {
        $this->confirmAction(
            'Approve Annual Results?',
            'This will approve the annual results for the selected students. you may not be able to undo.',
            'Yes, Approve',
            'Cancel'
        );
    }

    protected $listeners = ['action-confirmed' => 'approveAnnualResults'];

    public function approveAnnualResults()
    {
        $this->validate([
            'selectedStudents' => 'required|array|min:1',
            'selectedYear' => 'required',
            'punctuality' => 'required|numeric|min:0|max:5',
            'mental_alertness' => 'required|numeric|min:0|max:5',
            'attentiveness' => 'required|numeric|min:0|max:5',
            'respect' => 'required|numeric|min:0|max:5',
            'neatness' => 'required|numeric|min:0|max:5',
            'politeness' => 'required|numeric|min:0|max:5',
            'honesty' => 'required|numeric|min:0|max:5',
            'relationship_with_peers' => 'required|numeric|min:0|max:5',
            'attitude_to_school' => 'required|numeric|min:0|max:5',
            'teamwork' => 'required|numeric|min:0|max:5',
            'completes_school_work_promptly' => 'required|numeric|min:0|max:5',
            'reading' => 'required|numeric|min:0|max:5',
            'verbal_fluency_diction' => 'required|numeric|min:0|max:5',
            'handwriting' => 'required|numeric|min:0|max:5',
            'musical_skills' => 'required|numeric|min:0|max:5',
            'creative_arts' => 'required|numeric|min:0|max:5',
            'physical_education' => 'required|numeric|min:0|max:5',
            'general_reasoning' => 'required|numeric|min:0|max:5',
            'principal_remarks' => 'required|string',
        ]);

        $approvalData = [
            'punctuality' => $this->punctuality,
            'mental_alertness' => $this->mental_alertness,
            'attentiveness' => $this->attentiveness,
            'respect' => $this->respect,
            'neatness' => $this->neatness,
            'politeness' => $this->politeness,
            'honesty' => $this->honesty,
            'relationship_with_peers' => $this->relationship_with_peers,
            'attitude_to_school' => $this->attitude_to_school,
            'teamwork' => $this->teamwork,
            'completes_school_work_promptly' => $this->completes_school_work_promptly,
            'reading' => $this->reading,
            'verbal_fluency_diction' => $this->verbal_fluency_diction,
            'handwriting' => $this->handwriting,
            'musical_skills' => $this->musical_skills,
            'creative_arts' => $this->creative_arts,
            'physical_education' => $this->physical_education,
            'general_reasoning' => $this->general_reasoning,
            'principal_remarks' => $this->principal_remarks,
        ];

        $annualResultService = new AnnualResultService();
        $annualResultService->approveAnnualResults(
            $this->selectedStudents,
            $this->selectedYear,
            $approvalData,
            Auth::id()
        );

        $this->showToast('success', 'Annual results approved successfully!', 'Annual Results Approval');

        // Refresh students list to exclude newly approved students
        $this->updatedSelectedClass();

        // Clear student results and selected students
        $this->selectedStudents = [];
        $this->studentResults = [];
    }

    public function render()
    {
        return view('livewire.results.annual-approval-form');
    }
}
