<?php

namespace App\Livewire\Results;

use Livewire\Component;
use App\Services\AnnualResultService;
use App\Traits\HasNotifications;

class AnnualMasterList extends Component
{
    use HasNotifications;

    public $year;
    public $classId;
    public $batchId;
    public $classes = [];
    public $batches = [];
    public $years = [];
    public $masterListData = [];
    public $subjects = [];
    public $sessions = [];
    public $selectedYear;
    public $selectedClass;
    public $selectedBatch;

    protected $annualResultService;

    public function boot(AnnualResultService $annualResultService)
    {
        $this->annualResultService = $annualResultService;
    }

    public function mount()
    {
        $this->classes = $this->annualResultService->getClassesForUser(auth()->user());
        $this->years = $this->annualResultService->getAvailableYears();
    }

    public function updatedClassId()
    {
        $this->batchId = null;
        $this->batches = [];
        $this->masterListData = [];

        if ($this->classId) {
            $this->batches = \App\Models\Batch::where('school_class_id', $this->classId)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        }
    }

    public function loadAnnualMasterList()
    {
        if (!$this->year || !$this->classId) {
            $this->showError('Please select both year and class');
            return;
        }

        $data = $this->annualResultService->getAnnualMasterListData($this->year, $this->classId, $this->batchId);

        if (!$data) {
            $this->showError('No data found for the selected criteria');
            return;
        }

        $this->masterListData = $data['masterListData'];
        $this->subjects = $data['subjects'];
        $this->sessions = $data['sessions'];
        $this->selectedYear = $data['year'];
        $this->selectedClass = $data['class'];
        $this->selectedBatch = $data['batch'];

        $this->dispatch('initMasterListChart', $this->getChartData());
        $this->showSuccess('Annual master list loaded successfully');
    }

    private function getChartData()
    {
        if (empty($this->masterListData)) {
            return [];
        }

        $studentNames = [];
        $annualAverages = [];
        
        foreach ($this->masterListData as $student) {
            $studentNames[] = $student['student']->firstname . ' ' . substr($student['student']->surname ?? '', 0, 1) . '.';
            $annualAverages[] = $student['annual_average'];
        }

        return [
            'labels' => $studentNames,
            'data' => $annualAverages
        ];
    }

    public function getPrintUrl()
    {
        if (!$this->selectedYear || !$this->selectedClass) {
            return null;
        }

        return route('results.print-annual-master-list', [
            'class' => $this->selectedClass->id,
            'year' => $this->selectedYear,
            'batch' => $this->selectedBatch ? $this->selectedBatch->id : null
        ]);
    }

    public function render()
    {
        return view('livewire.results.annual-master-list');
    }
}