<?php

namespace App\Livewire;

use App\Models\SchoolSession;
use App\Services\SchoolSessionService;
use App\Traits\HasNotifications;
use Livewire\Component;
use Livewire\WithPagination;

class SchoolSessionsTable extends Component
{
    use WithPagination, HasNotifications;

    protected $paginationTheme = 'bootstrap';

    protected $listeners = ['sessionDeleted' => '$refresh', 'action-confirmed' => 'handleConfirmedAction'];

    public $pendingAction = null;
    public $pendingSessionId = null;

    public function confirmActivateSession($sessionId)
    {
        $this->pendingAction = 'activate';
        $this->pendingSessionId = $sessionId;
        $this->confirmAction('Activate Session?', 'This will deactivate all other sessions and set this one as active.');
    }

    public function confirmDeleteSession($sessionId)
    {
        $this->pendingAction = 'delete';
        $this->pendingSessionId = $sessionId;
        $this->confirmAction('Delete Session?', 'This action cannot be undone!');
    }

    public function handleConfirmedAction()
    {
        if ($this->pendingAction === 'activate') {
            $this->activateSession($this->pendingSessionId);
        } elseif ($this->pendingAction === 'delete') {
            $this->deleteSession($this->pendingSessionId);
        }
        
        $this->pendingAction = null;
        $this->pendingSessionId = null;
    }

    public function activateSession($sessionId)
    {
        $session = SchoolSession::findOrFail($sessionId);
        $sessionService = app(SchoolSessionService::class);
        $sessionService->activateSession($session);
        
        $this->showToast('success', 'School session activated successfully!');
        $this->dispatch('sessionActivated');
    }

    public function deleteSession($sessionId)
    {
        $session = SchoolSession::findOrFail($sessionId);
        $sessionService = app(SchoolSessionService::class);
        $sessionService->deleteSession($session);
        
        $this->showToast('success', 'School session deleted successfully!');
        $this->dispatch('sessionDeleted');
    }

    public function render()
    {
        $sessions = SchoolSession::orderBy('start_year', 'desc')->paginate(10);
        
        return view('livewire.school-sessions-table', [
            'sessions' => $sessions
        ]);
    }
}