<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\User;
use App\Services\PayrollService;
use App\Traits\HasNotifications;

class StaffSalaryForm extends Component
{
    use WithPagination, HasNotifications;

    public $selectedStaff = null;
    public $monthly_salary = '';
    public $bank_name = '';
    public $account_number = '';
    public $account_name = '';
    public $tax_id = '';
    public $payment_day = 31;
    public $search = '';

    protected $rules = [
        'monthly_salary' => 'required|numeric|min:0',
        'bank_name' => 'nullable|string|max:255',
        'account_number' => 'nullable|string|max:255',
        'account_name' => 'nullable|string|max:255',
        'tax_id' => 'nullable|string|max:255',
        'payment_day' => 'required|integer|min:1|max:31',
    ];

    public function selectStaff($userId)
    {
        $this->selectedStaff = User::find($userId);
        $this->monthly_salary = $this->selectedStaff->monthly_salary ?? '';
        $this->bank_name = $this->selectedStaff->bank_name ?? '';
        $this->account_number = $this->selectedStaff->account_number ?? '';
        $this->account_name = $this->selectedStaff->account_name ?? '';
        $this->tax_id = $this->selectedStaff->tax_id ?? '';
        $this->payment_day = $this->selectedStaff->payment_day ?? 31;
    }

    public function updateSalary()
    {
        $this->validate();

        try {
            $payrollService = new PayrollService();
            $payrollService->updateStaffSalary($this->selectedStaff, [
                'monthly_salary' => $this->monthly_salary,
                'bank_name' => $this->bank_name,
                'account_number' => $this->account_number,
                'account_name' => $this->account_name,
                'tax_id' => $this->tax_id,
                'payment_day' => $this->payment_day,
            ]);

            $this->showSuccess('Staff salary updated successfully!', 'Payroll Management');
            $this->reset(['selectedStaff', 'monthly_salary', 'bank_name', 'account_number', 'account_name', 'tax_id', 'payment_day']);
        } catch (\Exception $e) {
            $this->showError('Failed to update staff salary: ' . $e->getMessage(), 'Payroll Management');
        }
    }

    public function render()
    {
        $staff = User::whereHas('roles')
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('firstname', 'like', '%' . $this->search . '%')
                      ->orWhere('surname', 'like', '%' . $this->search . '%')
                      ->orWhere('email', 'like', '%' . $this->search . '%');
                });
            })
            ->with('roles')
            ->paginate(10);

        return view('livewire.staff-salary-form', compact('staff'));
    }
}