<?php

namespace App\Livewire;

use App\Models\User;
use App\Services\StudentService;
use App\Traits\HasNotifications;
use Livewire\Component;
use Livewire\WithPagination;

class StudentsTable extends Component
{
    use WithPagination, HasNotifications;

    public $search = '';
    public $perPage = 10;
    public $batchFilter = '';
    public $classFilter = '';
    public $statusFilter = 'active';
    protected $paginationTheme = 'bootstrap';

    public function mount()
    {
        $this->perPage = (int)$this->perPage;

        // Set default status filter to 'active' if not already set via URL
        if (!request()->has('statusFilter')) {
            $this->statusFilter = 'active';
        }
    }

    protected $queryString = ['search', 'batchFilter', 'classFilter', 'statusFilter'];
    protected $listeners = ['studentDeleted' => '$refresh'];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingBatchFilter()
    {
        $this->resetPage();
    }

    public function updatingClassFilter()
    {
        $this->batchFilter = '';
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }

    public function deleteStudent($id)
    {
        try {
            $student = User::findOrFail($id);
            $studentName = $student->name;

            app(StudentService::class)->deleteStudent($student);

            session()->flash('toast', [
                'type' => 'success',
                'message' => "Student {$studentName} deleted successfully"
            ]);

        } catch (\Exception $e) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Failed to delete student: ' . $e->getMessage()
            ]);
        }
    }

    public function toggleStudentStatus($studentId)
    {
        try {
            $student = User::findOrFail($studentId);
            $newStatus = !$student->active;
            $student->update(['active' => $newStatus]);

            $status = $newStatus ? 'activated' : 'deactivated';
            session()->flash('toast', [
                'type' => 'success',
                'message' => "Student {$status} successfully"
            ]);

        } catch (\Exception $e) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Failed to update student status: ' . $e->getMessage()
            ]);
        }
    }

    public function render()
    {
        $students = User::whereDoesntHave('roles')
            ->with(['batch.schoolClass'])
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('firstname', 'like', '%' . $this->search . '%')
                        ->orWhere('middlename', 'like', '%' . $this->search . '%')
                        ->orWhere('surname', 'like', '%' . $this->search . '%')
                        ->orWhere('student_id', 'like', '%' . $this->search . '%')
                        ->orWhere('email', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->classFilter, function ($query) {
                $query->whereHas('batch', function ($q) {
                    $q->where('school_class_id', $this->classFilter);
                });
            })
            ->when($this->batchFilter, function ($query) {
                $query->where('batch_id', $this->batchFilter);
            })
            ->when($this->statusFilter, function ($query) {
                if ($this->statusFilter === 'active') {
                    $query->where('active', true);
                } elseif ($this->statusFilter === 'inactive') {
                    $query->where('active', false);
                } elseif ($this->statusFilter === 'graduated') {
                    $query->whereHas('batch', function ($q) {
                        $q->where('graduated', true);
                    });
                }
            })
            ->when(!$this->batchFilter && $this->statusFilter !== 'graduated', function ($query) {
                $query->whereHas('batch', function ($q) {
                    $q->where('graduated', false);
                });
            })
            ->orderBy('firstname')
            ->paginate((int)$this->perPage);

        $classes = \App\Models\SchoolClass::orderBy('alias')->get();
        $batches = $this->classFilter
            ? \App\Models\Batch::where('school_class_id', $this->classFilter)->orderBy('name')->get()
            : \App\Models\Batch::orderBy('name')->get();

        return view('livewire.students-table', compact('students', 'batches', 'classes'));
    }
}
