<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Helpers\CurrencyHelper;

class Payment extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'payment_request_id',
        'payment_id',
        'user_id',
        'status',
        'amount',
        'currency',
        'notes',
        'recorded_by',
        'payment_date',
    ];
    
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($payment) {
            if (empty($payment->payment_id)) {
                $payment->payment_id = static::generateUniquePaymentId();
            }
        });
    }
    
    private static function generateUniquePaymentId()
    {
        do {
            $id = strtoupper(substr(str_shuffle('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 10));
        } while (static::where('payment_id', $id)->exists());
        
        return $id;
    }

    protected $casts = [
        'amount' => 'decimal:2',
        'payment_date' => 'datetime',
    ];

    public function paymentRequest()
    {
        return $this->belongsTo(PaymentRequest::class);
    }

    public function recordedBy()
    {
        return $this->belongsTo(User::class, 'recorded_by');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    public function getAmountAttribute($value)
    {
        return (int) $value;
    }
    
    public function getCurrencySymbolAttribute()
    {
        return CurrencyHelper::getSymbol($this->currency);
    }
    
    public function getRouteKeyName()
    {
        return 'payment_id';
    }
}
