<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use App\Models\Result;
use App\Models\ResultApproval;

class SchoolSession extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'slug',
        'start_year',
        'end_year',
        'start_date',
        'end_date',
        'next_session_start_date',
        'is_active',
        'description',
        'ca_max_score',
        'exam_max_score',
        'ca_pass_mark',
        'exam_pass_mark',
        'grade_a_min',
        'grade_b_min',
        'grade_c_min',
        'grade_d_min',
        'grade_e_min',
        'grade_f_min',
        'overall_pass_percentage',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'next_session_start_date' => 'date',
        'is_active' => 'boolean',
        'start_year' => 'integer',
        'end_year' => 'integer',
        'ca_max_score' => 'integer',
        'exam_max_score' => 'integer',
        'ca_pass_mark' => 'integer',
        'exam_pass_mark' => 'integer',
        'grade_a_min' => 'integer',
        'grade_b_min' => 'integer',
        'grade_c_min' => 'integer',
        'grade_d_min' => 'integer',
        'grade_e_min' => 'integer',
        'grade_f_min' => 'integer',
        'overall_pass_percentage' => 'integer',
    ];

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByYear($query, $year)
    {
        return $query->where('start_year', '<=', $year)
                    ->where('end_year', '>=', $year);
    }

    /**
     * Result approvals relationship
     */
    public function resultApprovals()
    {
        return $this->hasMany(ResultApproval::class, 'session_id');
    }

    /**
     * Results relationship
     */
    public function results()
    {
        return $this->hasMany(Result::class, 'session_id');
    }

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($session) {
            if (empty($session->slug)) {
                $session->slug = Str::slug($session->name) . '-' . Str::random(3);
            }
        });
    }
}