<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    use SoftDeletes;

    protected $fillable = ['key', 'value'];

    public static function get($key, $default = null)
    {
        $value = Cache::remember("setting_{$key}", 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        });

        return self::castValue($key, $value);
    }

    private static function castValue($key, $value)
    {
        $casts = [
            'annual_pass_percentage' => 'float',
            'show_class_teachers_on_print' => 'boolean',
            'show_class_teachers_online' => 'boolean',
            'show_timestamp' => 'boolean',
            'show_proprietor_signature' => 'boolean',
            'show_head_teacher_signature' => 'boolean',
            'allow_teacher_result_upload' => 'boolean',
            'allow_admin_result_upload' => 'boolean',
            'allow_teacher_edit_approved_results' => 'boolean',
            'allow_admin_edit_approved_results' => 'boolean',
            'allow_head_teacher_edit_approved_results' => 'boolean',
            'allow_teacher_unapprove_results' => 'boolean',
            'allow_admin_unapprove_results' => 'boolean',
            'allow_head_teacher_unapprove_results' => 'boolean',
            'allow_teacher_approve_results' => 'boolean',
            'allow_head_teacher_approve_results' => 'boolean',
            'allow_admin_approve_results' => 'boolean',
            'allow_staff_salary_view' => 'boolean',
        ];

        if (!isset($casts[$key]) || $value === null) {
            return $value;
        }

        return match ($casts[$key]) {
            'int', 'integer' => (int) $value,
            'float', 'double' => (float) $value,
            'bool', 'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'array' => json_decode($value, true),
            'object' => json_decode($value),
            default => $value,
        };
    }

    public static function set($key, $value)
    {
        $setting = self::updateOrCreate(['key' => $key], ['value' => $value]);
        Cache::forget("setting_{$key}");
        return $setting;
    }

    protected static function boot()
    {
        parent::boot();

        static::updated(function ($setting) {
            Cache::forget("setting_{$setting->key}");
        });

        static::created(function ($setting) {
            Cache::forget("setting_{$setting->key}");
        });
    }
}