<?php

namespace App\Services;

use App\Models\Result;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\Subject;
use App\Models\Batch;
use App\Models\SchoolSession;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class OfflineResultService
{
    public function generateTemplate($batchId, $sessionId)
    {
        $batch = Batch::with(['schoolClass.subjects', 'schoolClass.subjectAliases'])->find($batchId);
        $students = User::with(['results' => function($query) use ($sessionId) {
            $query->where('session_id', $sessionId);
        }])->where('batch_id', $batchId)->where('active', true)->orderBy('firstname')->get();
        $subjects = $batch->schoolClass->subjects;
        $session = SchoolSession::find($sessionId);
        
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set headers
        $headers = ['Student ID', 'Student Name'];
        $col = 3;
        
        foreach ($subjects as $subject) {
            $alias = $batch->schoolClass->subjectAliases->where('subject_id', $subject->id)->first();
            $displayName = $alias && $alias->alias ? $alias->alias : $subject->name;
            
            $headers[] = $displayName . ' - CA';
            $headers[] = $displayName . ' - Exam';
            $headers[] = $displayName . ' - Total';
            $headers[] = $displayName . ' - CA Absent';
            $headers[] = $displayName . ' - Exam Absent';
            $col += 5;
        }
        
        // Write headers
        for ($i = 0; $i < count($headers); $i++) {
            $cellCoordinate = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($i + 1) . '1';
            $sheet->setCellValue($cellCoordinate, $headers[$i]);
            $sheet->getStyle($cellCoordinate)->getFont()->setBold(true);
            $sheet->getStyle($cellCoordinate)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FFE0E0E0');
        }
        
        // Write student data
        $row = 2;
        foreach ($students as $student) {
            $sheet->setCellValue('A' . $row, $student->student_id);
            $sheet->setCellValue('B' . $row, $student->name);
            
            $col = 3;
            foreach ($subjects as $subject) {
                $caCell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col) . $row;
                $examCell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 1) . $row;
                $totalCell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 2) . $row;
                $caAbsentCell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 3) . $row;
                $examAbsentCell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 4) . $row;
                
                // Check for existing result
                $existingResult = Result::where('student_id', $student->id)
                    ->where('session_id', $sessionId)
                    ->where('subject_id', $subject->id)
                    ->first();
                
                // Set values (existing or default)
                $caScore = $existingResult ? $existingResult->ca_score : 0;
                $examScore = $existingResult ? $existingResult->exam_score : 0;
                $caAbsent = $existingResult ? $existingResult->ca_absent : 0;
                $examAbsent = $existingResult ? $existingResult->exam_absent : 0;
                
                $sheet->setCellValue($caCell, $caScore);
                $sheet->setCellValue($examCell, $examScore);
                $sheet->setCellValue($totalCell, "=$caCell+$examCell");
                $sheet->setCellValue($caAbsentCell, $caAbsent);
                $sheet->setCellValue($examAbsentCell, $examAbsent);
                
                // Add validation for CA score
                $caValidation = $sheet->getCell($caCell)->getDataValidation();
                $caValidation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_WHOLE);
                $caValidation->setOperator(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::OPERATOR_BETWEEN);
                $caValidation->setFormula1('0');
                $caValidation->setFormula2($session->ca_max_score);
                $caValidation->setShowErrorMessage(true);
                $caValidation->setErrorTitle('Invalid CA Score');
                $caValidation->setError('CA score must be between 0 and ' . $session->ca_max_score);
                
                // Add validation for Exam score
                $examValidation = $sheet->getCell($examCell)->getDataValidation();
                $examValidation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_WHOLE);
                $examValidation->setOperator(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::OPERATOR_BETWEEN);
                $examValidation->setFormula1('0');
                $examValidation->setFormula2($session->exam_max_score);
                $examValidation->setShowErrorMessage(true);
                $examValidation->setErrorTitle('Invalid Exam Score');
                $examValidation->setError('Exam score must be between 0 and ' . $session->exam_max_score);
                
                // Add validation for CA Absent (0 or 1)
                $caAbsentValidation = $sheet->getCell($caAbsentCell)->getDataValidation();
                $caAbsentValidation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
                $caAbsentValidation->setFormula1('"0,1"');
                $caAbsentValidation->setShowErrorMessage(true);
                $caAbsentValidation->setErrorTitle('Invalid Value');
                $caAbsentValidation->setError('Only 0 or 1 allowed');
                
                // Add validation for Exam Absent (0 or 1)
                $examAbsentValidation = $sheet->getCell($examAbsentCell)->getDataValidation();
                $examAbsentValidation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
                $examAbsentValidation->setFormula1('"0,1"');
                $examAbsentValidation->setShowErrorMessage(true);
                $examAbsentValidation->setErrorTitle('Invalid Value');
                $examAbsentValidation->setError('Only 0 or 1 allowed');
                
                $col += 5;
            }
            $row++;
        }
        
        // Auto-size columns
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($sheet->getHighestColumn());
        for ($i = 1; $i <= $highestColumnIndex; $i++) {
            $col = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($i);
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $writer = new Xlsx($spreadsheet);
        $filename = 'results_template_' . $batch->name . '_' . $session->name . '_' . date('Y-m-d_H-i-s') . '.xlsx';
        $filepath = storage_path('app/temp/' . $filename);
        
        // Add metadata sheet for validation
        $metaSheet = $spreadsheet->createSheet();
        $metaSheet->setTitle('METADATA');
        $metaSheet->setCellValue('A1', 'BATCH_ID');
        $metaSheet->setCellValue('B1', $batchId);
        $metaSheet->setCellValue('A2', 'SESSION_ID');
        $metaSheet->setCellValue('B2', $sessionId);
        $metaSheet->setCellValue('A3', 'BATCH_NAME');
        $metaSheet->setCellValue('B3', $batch->name);
        $metaSheet->setCellValue('A4', 'SESSION_NAME');
        $metaSheet->setCellValue('B4', $session->name);
        
        // Hide metadata sheet
        $metaSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        
        if (!file_exists(dirname($filepath))) {
            mkdir(dirname($filepath), 0755, true);
        }
        
        $writer->save($filepath);
        
        return [
            'filepath' => $filepath,
            'filename' => $filename
        ];
    }
    
    public function processUpload($filePath, $batchId, $sessionId)
    {
        $batch = Batch::with(['schoolClass.subjects', 'schoolClass.subjectAliases'])->find($batchId);
        $subjects = $batch->schoolClass->subjects;
        
        $spreadsheet = IOFactory::load($filePath);
        
        // Validate metadata if exists
        try {
            $metaSheet = $spreadsheet->getSheetByName('METADATA');
            if ($metaSheet) {
                $templateBatchId = $metaSheet->getCell('B1')->getValue();
                $templateSessionId = $metaSheet->getCell('B2')->getValue();
                
                if ($templateBatchId != $batchId) {
                    throw new \Exception('Template was generated for a different batch. Please download a new template.');
                }
                
                if ($templateSessionId != $sessionId) {
                    throw new \Exception('Template was generated for a different session. Please download a new template.');
                }
            }
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), 'Template was generated') !== false) {
                throw $e;
            }
            // If metadata sheet doesn't exist, continue (backward compatibility)
        }
        
        $sheet = $spreadsheet->getActiveSheet();
        $highestRow = $sheet->getHighestRow();
        
        $results = [
            'updated' => 0,
            'ignored' => 0,
            'wrong_class' => 0,
            'errors' => []
        ];
        
        DB::transaction(function () use ($sheet, $highestRow, $subjects, $sessionId, $batchId, &$results) {
            for ($row = 2; $row <= $highestRow; $row++) {
                $studentId = $sheet->getCell('A' . $row)->getValue();
                
                if (!$studentId) continue;
                
                $student = User::where('student_id', $studentId)->first();
                if (!$student) {
                    $results['errors'][] = "Student ID $studentId not found at row $row";
                    continue;
                }
                
                // Check if student belongs to the selected batch
                if ($student->batch_id != $batchId) {
                    $results['wrong_class']++;
                    $results['errors'][] = "Student $studentId ({$student->name}) at row $row belongs to a different class/batch";
                    continue;
                }
                
                // Check if student has approved results
                $hasApprovedResults = Result::where('student_id', $student->id)
                    ->where('session_id', $sessionId)
                    ->where('approved', true)
                    ->exists();
                
                if ($hasApprovedResults) {
                    $results['ignored']++;
                    continue;
                }
                
                $col = 3;
                foreach ($subjects as $subject) {
                    $caScore = $sheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col) . $row)->getValue() ?? 0;
                    $examScore = $sheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 1) . $row)->getValue() ?? 0;
                    $caAbsent = $sheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 3) . $row)->getValue() ?? 0;
                    $examAbsent = $sheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col + 4) . $row)->getValue() ?? 0;
                    
                    // Skip if total exceeds 100
                    if (($caScore + $examScore) > 100) {
                        $col += 5;
                        continue;
                    }
                    
                    Result::updateOrCreate(
                        [
                            'student_id' => $student->id,
                            'session_id' => $sessionId,
                            'subject_id' => $subject->id,
                        ],
                        [
                            'ca_score' => $caScore,
                            'exam_score' => $examScore,
                            'ca_absent' => $caAbsent,
                            'exam_absent' => $examAbsent,
                        ]
                    );
                    
                    $col += 5;
                }
                
                $results['updated']++;
            }
        });
        
        return $results;
    }
}